import express from 'express';
import multer from 'multer';
import { removeBackground } from '@imgly/background-removal-node';

const app = express();
const port = 3000;

app.use(express.json());

// Use memory storage so we don't need to read/write temporary files to disk
const upload = multer({ storage: multer.memoryStorage() });

app.get('/', (req, res) => {
  res.json({ message: 'Welcome to the background removal API' });
});

app.get('/health', (req, res) => {
  res.json({ status: 'ok' });
});

app.post('/api/remove-background', upload.single('image'), async (req, res) => {
  if (!req.file) {
    return res
      .status(400)
      .send('No image uploaded. Please upload a file with the key "image".');
  }

  try {
    // 1. Convert the Node.js Buffer to a standard Web Blob
    const imageBlob = new Blob([req.file.buffer], { type: req.file.mimetype });

    // Define the configuration for WebP output
    const config = {
      output: {
        format: 'image/webp',
        quality: 0.8, // Optional: Adjust quality between 0.0 and 1.0
      },
    };

    // 2. Pass the Blob to the background removal model
    const resultBlob = await removeBackground(imageBlob, config);

    // 3. Convert the resulting Blob back to a Node Buffer
    const arrayBuffer = await resultBlob.arrayBuffer();
    const buffer = Buffer.from(arrayBuffer);

    // 4. Send the new image back to the client
    res.set('Content-Type', 'image/webp');
    res.send(buffer);
  } catch (error) {
    console.error('Processing error:', error);
    res.status(500).send('Failed to process the image.');
  }
});

app.listen(port, () => {
  console.log(`Server running at http://localhost:${port}`);
});
